<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class ErrorLog extends Model
{
    use HasFactory;

    protected $fillable = [
        'level',
        'type',
        'message',
        'file',
        'line',
        'trace',
        'url',
        'method',
        'ip',
        'user_agent',
        'user_id',
        'context',
        'is_resolved',
        'is_sent',
        'sent_at',
    ];

    protected $casts = [
        'context' => 'array',
        'is_resolved' => 'boolean',
        'is_sent' => 'boolean',
        'sent_at' => 'datetime',
        'created_at' => 'datetime',
        'updated_at' => 'datetime',
    ];

    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    /**
     * Scope for unresolved errors
     */
    public function scopeUnresolved($query)
    {
        return $query->where('is_resolved', false);
    }

    /**
     * Scope for resolved errors
     */
    public function scopeResolved($query)
    {
        return $query->where('is_resolved', true);
    }

    /**
     * Scope for critical errors
     */
    public function scopeCritical($query)
    {
        return $query->where('level', 'critical');
    }

    /**
     * Scope for errors by level
     */
    public function scopeByLevel($query, $level)
    {
        return $query->where('level', $level);
    }

    /**
     * Get short message (first 100 chars)
     */
    public function getShortMessageAttribute(): string
    {
        return \Illuminate\Support\Str::limit($this->message, 100);
    }

    /**
     * Get formatted file location
     */
    public function getFileLocationAttribute(): string
    {
        if (!$this->file) {
            return 'Unknown';
        }
        
        $file = str_replace(base_path(), '', $this->file);
        return $file . ($this->line ? ':' . $this->line : '');
    }

    /**
     * Get level badge color
     */
    public function getLevelColorAttribute(): string
    {
        return match($this->level) {
            'critical' => 'danger',
            'error' => 'danger',
            'warning' => 'warning',
            'info' => 'info',
            default => 'secondary'
        };
    }

    /**
     * Mark as resolved
     */
    public function markAsResolved(): bool
    {
        return $this->update(['is_resolved' => true]);
    }

    /**
     * Mark as sent to support
     */
    public function markAsSent(): bool
    {
        return $this->update([
            'is_sent' => true,
            'sent_at' => now(),
        ]);
    }
}